/* COPYRIGHT 2000 by 3rd-evolution
 * Author: Thomas Krammer (tkrammer@3rd-evolution.de)
 */

#include "pch.h"
#include "Color.h"
#include "SerialDisplay.h"
#include "DialogBase.h"
#include "LCDSpectrum_Plugin.h"

// ==== globals ====

const char * const SETTINGS_DIRECTORY_NAME	=	"CL-Amp";
const char * const SETTINGS_FILE_NAME		=	"VisPlugin_LCDSpectrum.preferences";

const char * const PREF_DISPLAY_WIDTH		= 	"displayWidth";
const char * const PREF_DISPLAY_HEIGHT		= 	"displayHeight";
const char * const PREF_SERIAL_DEVICE		=	"serialDevice";
const char * const PREF_DISPLAY_SONG_INFO	=	"displaySongInfo";

// ==== entry point ====

VisualizationPlugin* NewPlugin(int *Version)
{
	*Version= CURRENT_VISUALIZATION_PLUGIN_VERSION;

	return new CLCDSpectrum_Plugin();
}

// ==== CLCDSpectrum_Plugin ====

CLCDSpectrum_Plugin::CLCDSpectrum_Plugin() :
	VisualizationPlugin("LCD Spectrum", "LCD Spectrum")
{
	serialDisplay	= NULL;
	spectrumData	= NULL;
	frontEnd		= NULL;
	
	displayWidth = displayHeight = 0;

	displaySongInfo	= false;
	
	scrollDelay	= 500000;
	
	renderThread = -1;
	
	exitThread = false;
	
	ReadSettings();
}

CLCDSpectrum_Plugin::~CLCDSpectrum_Plugin()
{
	if(renderThread >= 0) {
		// tell thread to exit
		exitThread = true;
		
		status_t exitValue;
		
		wait_for_thread(renderThread, &exitValue);
	}

	delete serialDisplay;
	delete spectrumData;
	delete frontEnd;
	
	WriteSettings();
}

// Called by CL-Amp when this plugin is activated	
bool CLCDSpectrum_Plugin::Init()
{
	Lock();

	if(serialDisplay == NULL) {
		SetSerialDevice(serialDevice.String());	
		SetDisplayWidth(displayWidth);
		SetDisplayHeight(displayHeight);
	}

	if(frontEnd == NULL) {
		frontEnd = new CLAmp_FrontEnd(true, scrollDelay);
	}

	frontEnd->StartRunning();

	if(renderThread < 0) {
		renderThread = spawn_thread(ThreadStartFunc, "LCDRenderThread", 
			B_NORMAL_PRIORITY, (void *)this);
			
		if(renderThread >= 0)
			resume_thread(renderThread);
	}
	
	Unlock();
	
	return serialDisplay && serialDisplay->InitCheck() == B_OK && 
		renderThread >= 0;
}

// Called by CL-Amp when this plugin is deactivated
void CLCDSpectrum_Plugin::Cleanup()
{
	if(renderThread >= 0) {
		// tell thread to exit
		exitThread = true;
		
		status_t exitValue;
		
		wait_for_thread(renderThread, &exitValue);
		
		renderThread = -1;
		exitThread = false;
	}

	Lock();

	if(frontEnd)
		frontEnd->StopRunning();

	// clear display
	if(serialDisplay)
		serialDisplay->Clear();
		
	Unlock();
}

bool CLCDSpectrum_Plugin::About(bool Question)
{
	if(!Question) {
		BAlert *alert = new BAlert( "LCDSpectrumPlugin_About",
									"LCDSpectrum Plugin\n"
									"Version 1.1\n\n"
									B_UTF8_COPYRIGHT " 2000-2001 by Thomas Krammer\n"
									"E-mail: tkrammer@3rd-evolution.de\n"
									"http://www.3rd-evolution.de",
									"OK" );
		
		BTextView *alertTextView = alert->TextView();

		text_run_array *textRunArray = 
			reinterpret_cast<text_run_array *>(new int8[sizeof(text_run_array) + sizeof(text_run)]);

		BFont bigFont = *be_plain_font;
		bigFont.SetSize(20);

		textRunArray->count			 = 2;
		textRunArray->runs[0].offset = 0;
		textRunArray->runs[0].font	 = bigFont;
		textRunArray->runs[0].color	 = CColor::Black;
		textRunArray->runs[1].offset = 19;
		textRunArray->runs[1].font	 = *be_plain_font;
		textRunArray->runs[1].color	 = CColor::Black;

		alertTextView->SetStylable(true);
		alertTextView->SetRunArray(0, alertTextView->TextLength(), textRunArray);
		
		alert->SetShortcut(0, B_ESCAPE);
		alert->ChildAt(0)->ResizeToPreferred();
		alert->ResizeTo(alert->ChildAt(0)->Bounds().Width(), alert->ChildAt(0)->Bounds().Height());
		alert->Go();
		
		delete textRunArray;
	}
	
	return true;
}

bool CLCDSpectrum_Plugin::Prefs(bool Question)
{
	if(!Question) {
		BView *prefsView = new CPluginPrefView(BRect(0,0,50,50), this);
	
		float windowWidth, windowHeight;
	
		prefsView->GetPreferredSize(&windowWidth, &windowHeight);
	
		BRect screenRect		= BScreen().Frame();
		BPoint windowLeftTop	= BPoint((screenRect.Width()-windowWidth)/2, 
									(screenRect.Height()-windowHeight)/2);
	
		BRect windowRect(windowLeftTop, windowLeftTop + BPoint(windowWidth, windowHeight));
	
		BWindow *perfsWindow = new BWindow(windowRect, "LCD Settings", 
			B_TITLED_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL, B_NOT_ZOOMABLE | B_NOT_RESIZABLE);
		
		prefsView->ResizeTo(windowWidth, windowHeight);
			
		perfsWindow->AddChild(prefsView);
		perfsWindow->Show();
	}
	
	return true;
}

unsigned long CLCDSpectrum_Plugin::GetFlags()
{
	return VISPLUG_NEED_AUDIO_DATA;
}

bool CLCDSpectrum_Plugin::Render(const struct VisAudioInfoStruct *info)
{
	// This method only calculates the new values. The real rendering is done
	// in a own thread.

	Lock();

	bool result = false;

	if(serialDisplay) {
		// Calc new spectrum data for each column. Average of
		// 'barData' spectrum data values is value for one column.
	
		float barData = SPECTRUMDATA_LEN / displayWidth;
		const int8 falloff = 5;
	
		for(int32 col=0 ; col<displayWidth ; col++) {
			int32 value=0;
			
			for(int32 k=(int32)(col*barData) ; k<MIN((col+1)*barData, SPECTRUMDATA_LEN) ; k++) {
				for(int channel=0 ; channel < info->SpectrumChannels ; channel++) {
					value += info->SpectrumData[k+SPECTRUMDATA_LEN*channel];
				}
			}
			
			spectrumData[col] = (uint8)MAX(spectrumData[col]-falloff, 
				(value / (float)(barData*info->SpectrumChannels)));
		}
	
		result = true;
	}

	Unlock();
	
	return result;
}

int32 CLCDSpectrum_Plugin::ThreadStartFunc(void *data)
{
	reinterpret_cast<CLCDSpectrum_Plugin *>(data)->AsyncRender();
	return 0;
}

void CLCDSpectrum_Plugin::AsyncRender()
{
	bigtime_t lastScrollTime = 0;
	int scrollPos = 0;
	bool scrollReversed = false;
	
	long lastSongTitleLen = 0;

	// Buffer for a row. (255 should be enough for all displays...)
	char buffer[255];
	
	// This maps the number of pixels to draw in a bar to 
	// the (special) character to draw.
	char map[9] = { ' ', 1, 2, 3, 4, 5, 6, 7, 0 };

	while(!exitThread) {
		// I can lock the plugin while writing. This should NEVER block. The
		// write buffer is big enough to hold all the written data.
		Lock();

		int8 height = displaySongInfo ? displayHeight-1 : displayHeight;

		float mul		= (height*8) / 255.0;
		int32 dataRate	= data_rate_in_bps(serialDisplay->Port()->DataRate());

		// time in milliseconds needed to write display at a current data rate.
		// this is a rough estimation of the time, but every method must touch
		// at least every character of the display.
		// Very DrawVerticalBar() writes a four byte command sequence.
		bigtime_t wait	= (displayWidth * MAX(displayHeight, 4) * 8 * 1000000) / dataRate;

		if(displaySongInfo) {
			if(frontEnd->Refresh()) {
				const char *title = frontEnd->Title();
				int titleLen = strlen(title);
			
				if( titleLen != lastSongTitleLen ||
					(system_time() - lastScrollTime) >= scrollDelay) {
					// The title changed or it's time to scroll...
					
					if(titleLen != lastSongTitleLen )
						scrollPos = 0;
						
					serialDisplay->DisplayString(&title[scrollPos], 0, 0, displayWidth);
		
					if(titleLen < displayWidth) {
						for(int i=0 ; i<displayWidth-titleLen ; i++)
							serialDisplay->DisplayString(" ");
					} else {
						if(scrollReversed)
							scrollPos--;
						else
							scrollPos++;
					
						if(scrollPos >= titleLen-displayWidth || scrollPos < 0) {
							scrollReversed = !scrollReversed;
						}
					}
					
					lastScrollTime = system_time();
					lastSongTitleLen = titleLen;
				}
			}

			// I paint the bars myself, because the Draw[Vertical|Horizontal]Bar methods of
			// ISerialDisplay clear the whole column/row of the bar.
			// This would overwrite the title. (Reversing the order doesn't help)
			
			for(int32 j=1 ; j<displayHeight ; j++) {
				// index of the lowest pixel of this row.
				int sub = (displayHeight-j-1)*8;
	
				serialDisplay->CursorTo(0, j);
				
				for(int32 i=0 ; i<displayWidth ; i++) {
					int index = MAX(0, MIN((int)(spectrumData[i]*mul-sub), (int)sizeof(map)));
				
					buffer[i] = map[index];
				}
	
				serialDisplay->Write(buffer, displayWidth);	
			}
		} else {
			for(int32 i=0 ; i<displayWidth ; i++) {
				serialDisplay->DrawVerticalBar(i, (uchar)spectrumData[i]*mul);
			}
		}
		
		Unlock();
		
		// Give the display enough time to display the data.
		snooze(wait + 10000);
		
		Lock();
		// If the information still isn't written: get rid of it.
		serialDisplay->Port()->ClearOutput();
		Unlock();
	}
}

void CLCDSpectrum_Plugin::ReadSettings()
{
	BMessage settings;
	bool successful = false;

	BPath settingsFilePath;
	find_directory(B_USER_SETTINGS_DIRECTORY, &settingsFilePath);
	
	settingsFilePath.Append(SETTINGS_DIRECTORY_NAME);
	settingsFilePath.Append(SETTINGS_FILE_NAME);

	BFile settingsFile(settingsFilePath.Path(), B_READ_ONLY);

	if(settingsFile.InitCheck() == B_OK) {
		if(settings.Unflatten(&settingsFile) == B_OK) {
			settings.FindInt8(PREF_DISPLAY_WIDTH, &displayWidth);
			settings.FindInt8(PREF_DISPLAY_HEIGHT, &displayHeight);

			if(settings.FindBool(PREF_DISPLAY_SONG_INFO, &displaySongInfo) != B_OK) {
				// This value was added in version 1.1.
				// Maintain compatibilty with version 1.0.
				displaySongInfo = false;
			}

			serialDevice = settings.FindString(PREF_SERIAL_DEVICE);
		
			successful = true;
		}
	}
	
	if(!successful) {
		// fill in default values
		displayWidth	= 20;
		displayHeight	= 2;
		serialDevice	= "serial1";
		displaySongInfo	= false;
	}
}

void CLCDSpectrum_Plugin::WriteSettings()
{
	BMessage settings;
	
	settings.AddInt8(PREF_DISPLAY_WIDTH, displayWidth);
	settings.AddInt8(PREF_DISPLAY_HEIGHT, displayHeight);
	settings.AddString(PREF_SERIAL_DEVICE, serialDevice.String());
	settings.AddBool(PREF_DISPLAY_SONG_INFO, displaySongInfo);
	
	BPath settingsDirPath;
	find_directory(B_USER_SETTINGS_DIRECTORY, &settingsDirPath);
	
	settingsDirPath.Append(SETTINGS_DIRECTORY_NAME);
	
	// Create CL-Amp directory in settings dir, if it doesn't exist.
	BDirectory settingsDir;
	settingsDir.CreateDirectory(settingsDirPath.Path(), NULL);
	
	settingsDirPath.Append(SETTINGS_FILE_NAME);
	
	BFile settingsFile(settingsDirPath.Path(), B_WRITE_ONLY | B_CREATE_FILE);
	
	if(settingsFile.InitCheck() == B_OK)
		settings.Flatten(&settingsFile);
}

void CLCDSpectrum_Plugin::SetSerialDevice(const char *device)
{
	delete serialDisplay;

	serialDisplay = new CMatrixOrbitalSerialDisplay();
	
	status_t initResult = serialDisplay->Init(device);
	
	if(initResult == B_OK) {
		serialDisplay->AutoWrap(false);
		serialDisplay->AutoScroll(false);
		serialDisplay->Cursor(false, false);
		serialDisplay->Backlight(true);
		serialDisplay->Clear();
		serialDisplay->Home();
		serialDisplay->InitVerticalBarGraph();
		
		serialDevice = device;
	} else {
		BString message;
			
		message << "Can't open serial device '" << device << "'\n"
				<< "Reason: " << strerror(initResult);
		
		BAlert *alert = new BAlert("SerialError", message.String(), "OK");
			
		alert->Go();
	}
}

void CLCDSpectrum_Plugin::SetDisplayWidth(int8 width)
{
	delete spectrumData;
	
	displayWidth	= width;
	
	spectrumData = new uchar [displayWidth];
	
	memset(spectrumData, 0, sizeof(uchar)*displayWidth);
}

// ==== CPluginPrefView ====

const float CPluginPrefView::dist = 10.0;

CPluginPrefView::CPluginPrefView(BRect frame, CLCDSpectrum_Plugin *_plugin, uint32 resizingMode, uint32 flags) :
	CDialogBase(frame, "PerfView", resizingMode, flags)
{
	serialSel = sizeSel = NULL;
	
	plugin = _plugin;
	
	const char * const selDisplayInfoLabel	= "Display song title";

	plugin->Lock();

	selWidth		= plugin->DisplayWidth();
	selHeight		= plugin->DisplayHeight();
	selSerDevice	= plugin->SerialDevice();
	selDisplayInfo	= plugin->DisplaysSongInfo();
	
	plugin->Unlock();

	BRect dummy(0,0,20,20);
	displayInfoSel = new BCheckBox(dummy, "DisplayInfoSel", selDisplayInfoLabel, NULL, 
									B_FOLLOW_TOP | B_FOLLOW_LEFT_RIGHT);

	displayInfoSel->SetValue(selDisplayInfo ? B_CONTROL_ON : B_CONTROL_OFF);							
}

CPluginPrefView::~CPluginPrefView()
{
}

void CPluginPrefView::GetPreferredSize(float *width, float *height)
{
	float baseWidth, baseHeight;
	
	CDialogBase::GetPreferredSize(&baseWidth, &baseHeight);
	
	font_height fh;
	GetFontHeight(&fh);

	float displayInfoSelWidth, displayInfoSelHeight;
	
	displayInfoSel->GetPreferredSize(&displayInfoSelWidth, &displayInfoSelHeight);
	
	*width  = MAX(baseWidth, MAX(200, displayInfoSelWidth));
	*height = baseHeight + displayInfoSelHeight + 2 * (fh.ascent + fh.descent) + 5*dist;
}

void CPluginPrefView::AttachedToWindow()
{
	CDialogBase::AttachedToWindow();

	font_height fh;
	GetFontHeight(&fh);
	
	float fontHeight = fh.ascent + fh.descent;

	const char * const serialSelLabel		= "Port";
	const char * const sizeSelLabel			= "Size";

	float deviderPos = MAX( StringWidth(serialSelLabel), StringWidth(sizeSelLabel) ) + 10;

	BMenu *serialMenu = new BPopUpMenu("<Pick one>");
	
	BSerialPort serial;
	char devName[B_OS_NAME_LENGTH];
	
	for(int i=0 ; i<serial.CountDevices() ; i++) {
		serial.GetDeviceName(i, devName);
		
		BMessage *message = new BMessage(MSG_SER_DEVICE_SELECTED);
		message->AddString("device", devName);
		
		BMenuItem *item = new BMenuItem(devName, message);
		
		if(selSerDevice == devName)
			item->SetMarked(true);
		
		serialMenu->AddItem(item);
	}

	serialMenu->SetTargetForItems(this);

	BRect serialSelRect(dist, dist, Bounds().right-dist, dist+fontHeight);

	serialSel = new BMenuField(serialSelRect, "SerialSel", serialSelLabel, 
		serialMenu, B_FOLLOW_TOP | B_FOLLOW_LEFT_RIGHT);
		
	serialSel->SetDivider(deviderPos);
		
	BMenu *sizeMenu = new BPopUpMenu("<Pick one>");

	struct size_data {
		int8 width;
		int8 height;
		const char *label;
	} sizeData[] = {
		{ 20, 2, "20x2" },
		{ 40, 2, "40x2" },
		{ 20, 4, "20x4" },
		{ 40, 4, "40x4" },
	};

	for(int i=0 ; i<4 ; i++) {
		BMessage *message = new BMessage(MSG_SIZE_SELECTED);
		
		message->AddInt8("width", sizeData[i].width);
		message->AddInt8("height", sizeData[i].height);
		
		BMenuItem *item = new BMenuItem(sizeData[i].label, message);
		
		if(selWidth == sizeData[i].width && selHeight == sizeData[i].height) {
			item->SetMarked(true);
		}
		
		sizeMenu->AddItem(item);
	}	

	sizeMenu->SetTargetForItems(this);

	BRect sizeSelRect(dist, 2*dist+fontHeight, Bounds().right-dist, 2*(dist+fontHeight));

	sizeSel = new BMenuField(sizeSelRect, "SizeSel", sizeSelLabel, 
		sizeMenu, B_FOLLOW_TOP | B_FOLLOW_LEFT_RIGHT);
		
	sizeSel->SetDivider(deviderPos);

	displayInfoSel->MoveTo(dist, sizeSelRect.bottom + 2*dist);
	displayInfoSel->ResizeToPreferred();

	AddChild(serialSel);
	AddChild(sizeSel);
	AddChild(displayInfoSel);
}

void CPluginPrefView::MessageReceived(BMessage *message)
{
	switch(message->what) {
		case MSG_SER_DEVICE_SELECTED:
			{
				const char *device;
				
				if(message->FindString("device", &device) == B_OK) {
					selSerDevice = device;
				}
			}
			break;
		case MSG_SIZE_SELECTED:
			{
				message->FindInt8("width", &selWidth);
				message->FindInt8("height", &selHeight);
			}
			break;
		default:
			CDialogBase::MessageReceived(message);
	}
}

bool CPluginPrefView::Ok()
{
	plugin->Lock();

	plugin->SetDisplayWidth(selWidth);
	plugin->SetDisplayHeight(selHeight);
	plugin->SetSerialDevice(selSerDevice.String());
	plugin->SetDisplaySongInfo(displayInfoSel->Value() == B_CONTROL_ON);
	
	plugin->Unlock();

	return true;
}
