/* COPYRIGHT 2000 by 3rd-evolution
 * Author: Thomas Krammer (tkrammer@3rd-evolution.de)
 */

#include "pch.h"
#include "Color.h"
#include "FlickerFreeButton.h"
#include "DialogBase.h"

// ==== CDialogBase =====

// distance of the buttons from the border.
const float CDialogBase::dist = 10.0;

CDialogBase::CDialogBase(
	BRect frame,
	const char *name, 
	uint32 resizingMode,
	uint32 flags,
	const char *okButtonLabel,
	const char *cancelButtonLabel) :
	BView(frame, name, resizingMode, flags | B_FRAME_EVENTS)
{
	// intermediate position for the buttons during creation. They are
	// resized/moved afterwards.
	BRect dummyPos(0,0,5,5);

	// --- process 'OK' and 'Cancel' button
	
	// create buttons on dummy position

	const char *defaultOkLabel		= "OK";
	const char *defaultCancelLabel	= "Cancel";

	okButton = new CFlickerFreeButton(dummyPos, "OkButton", 
							okButtonLabel ? okButtonLabel : defaultOkLabel, 
							new BMessage(MSG_OK), 
							B_FOLLOW_BOTTOM | B_FOLLOW_RIGHT);
											
	cancelButton = new CFlickerFreeButton(dummyPos, "CancelButton", 
							cancelButtonLabel ? cancelButtonLabel : defaultCancelLabel, 
							new BMessage(MSG_CANCEL), 
							B_FOLLOW_BOTTOM | B_FOLLOW_RIGHT);

	// calculate real button size

	float okButtonHeight, okButtonWidth;
	float cancelButtonHeight, cancelButtonWidth;
	
	okButton->GetPreferredSize(&okButtonWidth, &okButtonHeight);
	cancelButton->GetPreferredSize(&cancelButtonWidth, &cancelButtonHeight);

	const float minButtonWidth	= 80.0;
	const float minButtonHeight	= 20.0;

	float buttonWidth  = MAX(okButtonWidth, MAX(cancelButtonWidth, minButtonWidth));
	float buttonHeight = MAX(okButtonHeight, MAX(cancelButtonHeight, minButtonHeight));

	okButton->ResizeTo(buttonWidth, buttonHeight);
	cancelButton->ResizeTo(buttonWidth, buttonHeight);

	// add buttons

	AddChild(okButton);
	AddChild(cancelButton);
}

CDialogBase::~CDialogBase()
{
	// the buttons are automatically deleted.
}

void CDialogBase::FrameResized(float width, float height)
{
	BView::FrameResized(width, height);

	// move buttons to correct position

	// A default button is resized to all sides by 3 pixel
	const float defaultButtonInset=-3;

	float buttonWidth  = cancelButton->Bounds().Width();
	float buttonHeight = cancelButton->Bounds().Height();
	
	BRect okButtonRect(width  - dist - buttonWidth,
					   height - dist - buttonHeight,
					   width  - dist,
					   height - dist);

	BRect cancelButtonRect = okButtonRect.OffsetByCopy(-(dist+buttonWidth), 0);

	if(okButton->IsDefault()) {
		okButtonRect.InsetBy(defaultButtonInset, defaultButtonInset);
	}

	okButton->MoveTo(okButtonRect.LeftTop());
	cancelButton->MoveTo(cancelButtonRect.LeftTop());
}

void CDialogBase::AttachedToWindow()
{
	BView::AttachedToWindow();

	SetViewColor(CColor::BeBackgroundGray);

	okButton->SetTarget(this);											
	cancelButton->SetTarget(this);
	
	// Simulate frame resized event to move buttons to 
	// correct position.
	FrameResized(Bounds().Width(), Bounds().Height());

	// install message hook for ESC keypress.
	// this class needs to be notified when the user presses ESC,
	// even if it's not the keyboard focus view.
	Window()->AddCommonFilter(new CEscapeMessageFilter(this));

	Window()->SetDefaultButton(okButton);
}

void CDialogBase::KeyDown(const char *bytes, int32 numBytes)
{
	switch(bytes[0]) {
		case B_ESCAPE:
			// I always get the ESC keypress (even when not keybord focus),
			// because I installed a message filter which redirects 
			// these messages.
			Cancel();
			Window()->PostMessage(B_QUIT_REQUESTED);
			break;
		default:
			BView::KeyDown(bytes, numBytes);
	}
}

void CDialogBase::MessageReceived(BMessage *message)
{
	switch(message->what) {
		case MSG_OK:
			if(Ok()) Window()->PostMessage(B_QUIT_REQUESTED);
			break;
		case MSG_CANCEL:
			Cancel();
			Window()->PostMessage(B_QUIT_REQUESTED);
			break;
	}
}

BRect CDialogBase::ClientRect()
{
	return BRect(
				0, 
				0, 
				Bounds().Width(), 
				Bounds().Height()-2*dist-cancelButton->Bounds().Height()
			);
}

void CDialogBase::GetPreferredSize(float *width, float *height)
{
	if(width) {
		*width = 3*dist + 2*(cancelButton->Bounds().Width());
	}

	if(height) {
		*height = 2*dist + cancelButton->Bounds().Height();
	}
}

// ==== CEscapeMessageFilter ====

CEscapeMessageFilter::CEscapeMessageFilter(BHandler *_escTarget) :
	BMessageFilter(B_ANY_DELIVERY, B_ANY_SOURCE, B_KEY_DOWN)
{
	escTarget = _escTarget;
}

// replaces the target for ESC keypress messages 'escTarget'.
filter_result CEscapeMessageFilter::Filter(
	BMessage *message, 
	BHandler **target
)
{
	if(message->what == B_KEY_DOWN) {
		int8 pressedChar;
	
		if(message->FindInt8("byte", 0, &pressedChar) == B_OK) {
			if(pressedChar == B_ESCAPE) {
				// replace target
				*target = escTarget;
			}
		}
	}
	
	// contiune processing
	return B_DISPATCH_MESSAGE;
}
